/**
 * Periodic Maintenance Configuration Controller
 */

const periodicMaintenanceConfigService = require('../services/periodicMaintenanceConfig.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult } = require('express-validator');

/**
 * Get all periodic maintenance configurations
 */
const getConfigs = async (req, res) => {
  try {
    const configs = await periodicMaintenanceConfigService.getConfigs(req.tenantId, req.query);
    return successResponse(res, configs, 'Configurations retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve configurations', 500);
  }
};

/**
 * Get configuration by ID
 */
const getConfigById = async (req, res) => {
  try {
    const { id } = req.params;
    const config = await periodicMaintenanceConfigService.getConfigById(id, req.tenantId);
    return successResponse(res, config, 'Configuration retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve configuration', 404);
  }
};

/**
 * Create new periodic maintenance configuration
 */
const createConfig = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const config = await periodicMaintenanceConfigService.createConfig(
      req.body,
      req.tenantId,
      req.userId
    );
    return successResponse(res, config, 'Configuration created successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create configuration', 400);
  }
};

/**
 * Update periodic maintenance configuration
 */
const updateConfig = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;
    const config = await periodicMaintenanceConfigService.updateConfig(
      id,
      req.body,
      req.tenantId
    );
    return successResponse(res, config, 'Configuration updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update configuration', 400);
  }
};

/**
 * Delete periodic maintenance configuration
 */
const deleteConfig = async (req, res) => {
  try {
    const { id } = req.params;
    const result = await periodicMaintenanceConfigService.deleteConfig(id, req.tenantId);
    return successResponse(res, result, 'Configuration deleted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to delete configuration', 400);
  }
};

/**
 * Get available products (products without periodic maintenance configuration)
 */
const getAvailableProducts = async (req, res) => {
  try {
    const products = await periodicMaintenanceConfigService.getAvailableProducts(req.tenantId, req.query);
    return successResponse(res, products, 'Available products retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve available products', 500);
  }
};

/**
 * Validation rules for creating a configuration
 */
const validateCreateConfig = [
  body('name').notEmpty().withMessage('Periodic Maintenance Name is required').trim(),
  body('productId').notEmpty().withMessage('Product selection is required').isMongoId().withMessage('Invalid product ID'),
  body('fields').isArray().withMessage('Fields must be an array'),
  body('fields.*.fieldType').isIn(['CHECKLIST', 'NUMBER_INPUT', 'RADIO_BUTTON']).withMessage('Invalid field type'),
  body('fields.*.fieldName').notEmpty().withMessage('Field name is required'),
  body('fields.*.isRequired').optional().isBoolean().withMessage('isRequired must be a boolean'),
];

/**
 * Validation rules for updating a configuration
 */
const validateUpdateConfig = [
  body('name').optional().notEmpty().withMessage('Periodic Maintenance Name cannot be empty').trim(),
  body('productId').optional().isMongoId().withMessage('Invalid product ID'),
  body('fields').optional().isArray().withMessage('Fields must be an array'),
];

module.exports = {
  getConfigs,
  getConfigById,
  createConfig,
  updateConfig,
  deleteConfig,
  getAvailableProducts,
  validateCreateConfig,
  validateUpdateConfig,
};

