/**
 * Periodic Maintenance Controller
 */

const periodicMaintenanceService = require('../services/periodicMaintenance.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult } = require('express-validator');

const getSchedules = async (req, res) => {
  try {
    const userRole = req.userRole;
    const userId = req.userId;
    // Get roleName from req.roleName (set by auth middleware) or from user object
    const roleName = req.roleName || req.user?.roleName || req.user?.roleId?.name || null;
    
    const schedules = await periodicMaintenanceService.getSchedules(
      req.tenantId,
      userRole,
      userId,
      roleName
    );
    return successResponse(res, schedules, 'Schedules retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve schedules', 500);
  }
};

const createSchedule = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const schedule = await periodicMaintenanceService.createSchedule(req.body, req.tenantId);
    return successResponse(res, schedule, 'Schedule created successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create schedule', 400);
  }
};

const generateJobs = async (req, res) => {
  try {
    const { scheduleId } = req.params;
    const jobs = await periodicMaintenanceService.generateJobsFromSchedule(scheduleId, req.tenantId);
    return successResponse(res, jobs, 'Jobs generated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to generate jobs', 400);
  }
};

const updateSchedule = async (req, res) => {
  try {
    const { id } = req.params;
    const schedule = await periodicMaintenanceService.updateSchedule(id, req.body, req.tenantId);
    return successResponse(res, schedule, 'Schedule updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update schedule', 400);
  }
};

const deleteSchedule = async (req, res) => {
  try {
    const { id } = req.params;
    const result = await periodicMaintenanceService.deleteSchedule(id, req.tenantId);
    return successResponse(res, result, 'Schedule deleted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to delete schedule', 400);
  }
};

const getScheduleJobs = async (req, res) => {
  try {
    const { scheduleId } = req.params;
    const Job = require('../models/Job.model');
    const jobService = require('../services/job.service');
    
    // Get jobs for this schedule
    const jobs = await Job.find({
      scheduleId,
      tenantId: req.tenantId
    })
      .populate('customerId', 'name email phone')
      .populate('assignedAgentId', 'name email phone')
      .populate('managerId', 'name email phone')
      .populate('productId', 'name')
      .sort({ createdAt: -1 });
    
    // Get timeline for each job
    const jobsWithTimeline = await Promise.all(
      jobs.map(async (job) => {
        try {
          const timeline = await jobService.getJobTimeline(job._id.toString(), req.tenantId);
          return {
            ...job.toObject(),
            timeline: timeline || []
          };
        } catch (error) {
          console.error(`Error fetching timeline for job ${job._id}:`, error);
          return {
            ...job.toObject(),
            timeline: []
          };
        }
      })
    );
    
    return successResponse(res, jobsWithTimeline, 'Schedule jobs retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve schedule jobs', 500);
  }
};

const getAgentSchedules = async (req, res) => {
  try {
    const schedules = await periodicMaintenanceService.getAgentSchedules(
      req.tenantId,
      req.userId,
      req.userRole,
      req.roleName
    );
    return successResponse(res, schedules, 'Agent schedules retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve agent schedules', 500);
  }
};

const getCustomerSchedules = async (req, res) => {
  try {
    const customerId = req.userId; // Customer's own ID
    const schedules = await periodicMaintenanceService.getCustomerSchedules(
      req.tenantId,
      customerId
    );
    return successResponse(res, schedules, 'Customer schedules retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve customer schedules', 500);
  }
};

const getScheduleById = async (req, res) => {
  try {
    const { id } = req.params;
    const schedule = await periodicMaintenanceService.getScheduleById(id, req.tenantId);
    return successResponse(res, schedule, 'Schedule retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve schedule', 500);
  }
};

const assignAgentToScheduleDate = async (req, res) => {
  try {
    const { id } = req.params;
    const { date, assignedAgentId } = req.body;
    
    if (!assignedAgentId) {
      return errorResponse(res, 'Agent ID is required', 400);
    }

    if (!date) {
      return errorResponse(res, 'Date is required', 400);
    }

    // Get agent name
    const User = require('../models/User.model');
    const Employee = require('../models/Employee.model');
    let agentName = null;

    const userAgent = await User.findOne({
      _id: assignedAgentId,
      tenantId: req.tenantId,
      role: 'SERVICE_AGENT',
    });

    if (userAgent) {
      agentName = userAgent.name;
    } else {
      const employee = await Employee.findById(assignedAgentId);
      if (employee) {
        agentName = `${employee.firstName} ${employee.lastName}`;
      } else {
        agentName = 'Unknown Agent';
      }
    }

    const schedule = await periodicMaintenanceService.assignAgentToScheduleDate(
      id,
      date,
      assignedAgentId,
      agentName,
      req.tenantId
    );
    
    return successResponse(res, schedule, 'Agent assigned to date successfully');
  } catch (error) {
    const logger = require('../utils/logger');
    logger.error('Assign agent to date error:', error);
    return errorResponse(res, error.message || 'Failed to assign agent to date', 400);
  }
};

const assignScheduleToAgent = async (req, res) => {
  try {
    const { id } = req.params;
    const { assignedAgentId } = req.body;
    
    if (!assignedAgentId) {
      return errorResponse(res, 'Agent ID is required', 400);
    }

    // Verify that the manager has access to this agent
    // Handle both Employee and User model managers
    const User = require('../models/User.model');
    const Employee = require('../models/Employee.model');
    const PeriodicMaintenanceSchedule = require('../models/PeriodicMaintenanceSchedule.model');
    const logger = require('../utils/logger');
    
    // Check if user is ADMIN or SUPER_ADMIN - if so, get manager from schedule
    const isAdmin = req.userRole === 'ADMIN' || req.userRole === 'SUPER_ADMIN';
    let manager = null;
    let scheduleManagerId = null;

    if (isAdmin) {
      // For admins, get the manager from the schedule itself
      logger.info('👑 [ASSIGN SCHEDULE] Admin user detected, getting manager from schedule...');
      const schedule = await PeriodicMaintenanceSchedule.findOne({ 
        _id: id, 
        tenantId: req.tenantId 
      });
      
      if (!schedule) {
        return errorResponse(res, 'Schedule not found', 404);
      }

      scheduleManagerId = schedule.managerId;
      
      if (scheduleManagerId) {
        // Get the manager from the schedule
        manager = await User.findOne({ 
          _id: scheduleManagerId, 
          tenantId: req.tenantId,
          role: 'MANAGER' 
        }).populate('assignedAgents', '_id name email');

        // If not found as User, check if it's an Employee with Manager role
        if (!manager) {
          logger.info('📱 [ASSIGN SCHEDULE] Manager not found as User, checking Employee collection...');
          const employee = await Employee.findById(scheduleManagerId).populate('roleId', 'name');
          
          if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'manager') {
            logger.info('✅ [ASSIGN SCHEDULE] Employee manager found, finding corresponding User account...');
            manager = await User.findOne({ 
              email: employee.email, 
              tenantId: req.tenantId,
              role: 'MANAGER' 
            }).populate('assignedAgents', '_id name email');
          }
        }
      }
    } else {
      // For managers, verify they are the manager
      manager = await User.findOne({ 
        _id: req.userId, 
        tenantId: req.tenantId,
        role: 'MANAGER' 
      }).populate('assignedAgents', '_id name email');

      // If not found as User, check if it's an Employee with Manager role
      if (!manager) {
        logger.info('📱 [ASSIGN SCHEDULE] Manager not found as User, checking Employee collection...');
        const employee = await Employee.findById(req.userId).populate('roleId', 'name');
        
        if (employee && employee.roleId && employee.roleId.name && employee.roleId.name.toLowerCase() === 'manager') {
          logger.info('✅ [ASSIGN SCHEDULE] Employee manager found, finding corresponding User account...');
          manager = await User.findOne({ 
            email: employee.email, 
            tenantId: req.tenantId,
            role: 'MANAGER' 
          }).populate('assignedAgents', '_id name email');
        }
      }
    }

    // If no manager found and not admin, return error
    if (!manager && !isAdmin) {
      return errorResponse(res, 'Manager not found', 403);
    }

    // For admins, if schedule has no manager or manager not found, allow assignment without manager verification
    if (isAdmin && !manager) {
      logger.info('👑 [ASSIGN SCHEDULE] Admin assigning agent without manager verification');
      // Skip manager verification for admins when manager is not found
    } else if (!manager) {
      return errorResponse(res, 'Manager not found', 403);
    }

    // For admins, skip manager agent verification if manager not found
    // For managers, verify they have assigned agents
    let agentExists = false;
    
    if (manager) {
      if (!manager.assignedAgents || manager.assignedAgents.length === 0) {
        // For admins, allow assignment even if manager has no agents
        if (!isAdmin) {
          return errorResponse(res, 'Manager has no assigned agents', 403);
        } else {
          logger.info('👑 [ASSIGN SCHEDULE] Admin assigning agent - manager has no assigned agents, skipping verification');
          agentExists = true; // Allow assignment for admins
        }
      } else {
        // Check if the agent is assigned to this manager
        // Handle both Employee IDs and User IDs
        agentExists = manager.assignedAgents.some(
          agent => (agent._id || agent.id).toString() === assignedAgentId.toString()
        );
      }
    } else {
      // For admins when manager not found, skip agent verification
      if (isAdmin) {
        logger.info('👑 [ASSIGN SCHEDULE] Admin assigning agent - no manager found, skipping agent verification');
        agentExists = true; // Allow assignment
      } else {
        // This should not happen as we already checked above, but keep for safety
        return errorResponse(res, 'Manager not found', 403);
      }
    }

    let finalAgentId = assignedAgentId;
    let finalAgentName = null;

    // If not found in assignedAgents, check if it's an Employee ID
    // and find the corresponding User account
    if (!agentExists && manager) {
      logger.info('📱 [ASSIGN SCHEDULE] Agent not found in assignedAgents, checking if it\'s an Employee ID...');
      const employee = await Employee.findById(assignedAgentId);
      if (employee) {
        // Find corresponding User account by email
        const userAgent = await User.findOne({
          email: employee.email,
          tenantId: req.tenantId,
          role: 'SERVICE_AGENT'
        });
        
        if (userAgent) {
          // Check if this User agent is in manager's assignedAgents
          const userAgentExists = manager.assignedAgents.some(
            agent => (agent._id || agent.id).toString() === userAgent._id.toString()
          );
          
          if (userAgentExists) {
            // Use the User agent ID instead
            finalAgentId = userAgent._id;
            finalAgentName = userAgent.name || `${employee.firstName} ${employee.lastName}`;
            agentExists = true;
            logger.info('✅ [ASSIGN SCHEDULE] Found corresponding User agent in manager\'s assignedAgents', {
              employeeId: assignedAgentId,
              userId: finalAgentId,
            });
          } else {
            // For admins, still use the agent even if not in manager's list
            if (isAdmin) {
              finalAgentId = userAgent._id;
              finalAgentName = userAgent.name || `${employee.firstName} ${employee.lastName}`;
              agentExists = true;
              logger.info('👑 [ASSIGN SCHEDULE] Admin assigning agent - User agent found but not in manager\'s list, allowing');
            } else {
              logger.warn('⚠️ [ASSIGN SCHEDULE] User agent found but not in manager\'s assignedAgents');
            }
          }
        } else {
          // For admins, still allow using Employee ID
          if (isAdmin) {
            finalAgentName = `${employee.firstName} ${employee.lastName}`;
            agentExists = true;
            logger.info('👑 [ASSIGN SCHEDULE] Admin assigning agent - Employee found, allowing assignment');
          } else {
            logger.warn('⚠️ [ASSIGN SCHEDULE] Employee found but no corresponding User account');
          }
        }
      }
      
      if (!agentExists && manager.assignedAgents) {
        // Also check if any assignedAgent is an Employee that matches
        const employee = await Employee.findById(assignedAgentId);
        if (employee) {
          // Check if any assigned agent (User) has the same email as this Employee
          for (const assignedAgent of manager.assignedAgents) {
            const userAgent = await User.findById(assignedAgent._id || assignedAgent.id);
            if (userAgent && userAgent.email === employee.email) {
              finalAgentId = userAgent._id;
              finalAgentName = userAgent.name || `${employee.firstName} ${employee.lastName}`;
              agentExists = true;
              logger.info('✅ [ASSIGN SCHEDULE] Found matching User agent by email', {
                employeeId: assignedAgentId,
                userId: finalAgentId,
              });
              break;
            }
          }
        }
      }
    }
    
    // Final check - for admins, allow assignment even if agent is not in manager's assigned agents
    if (!agentExists) {
      if (isAdmin) {
        logger.info('👑 [ASSIGN SCHEDULE] Admin assigning agent - agent not in manager\'s assigned agents, allowing assignment');
        agentExists = true; // Allow assignment for admins
        // Try to get agent name
        const userAgent = await User.findById(assignedAgentId);
        if (userAgent) {
          finalAgentName = userAgent.name;
        } else {
          const employeeAgent = await Employee.findById(assignedAgentId);
          if (employeeAgent) {
            finalAgentName = `${employeeAgent.firstName} ${employeeAgent.lastName}`;
          }
        }
      } else if (manager) {
        return errorResponse(res, 'Agent is not assigned to this manager', 403);
      }
    }

    // Get agent name if not already set
    if (!finalAgentName) {
      // First try to get from manager's assignedAgents if manager exists
      if (manager && manager.assignedAgents) {
        const agent = manager.assignedAgents.find(
          agent => (agent._id || agent.id).toString() === finalAgentId.toString()
        );
        finalAgentName = agent?.name;
      }
      
      // If still not found, try to get it from User or Employee
      if (!finalAgentName) {
        const userAgent = await User.findById(finalAgentId);
        if (userAgent) {
          finalAgentName = userAgent.name;
        } else {
          const employeeAgent = await Employee.findById(finalAgentId);
          if (employeeAgent) {
            finalAgentName = `${employeeAgent.firstName} ${employeeAgent.lastName}`;
          } else {
            finalAgentName = 'Unknown Agent';
          }
        }
      }
    }

    const schedule = await periodicMaintenanceService.assignScheduleToAgent(
      id,
      finalAgentId,
      finalAgentName,
      req.tenantId
    );
    
    return successResponse(res, schedule, 'Schedule assigned to agent successfully');
  } catch (error) {
    const logger = require('../utils/logger');
    logger.error('Assign schedule to agent error:', error);
    return errorResponse(res, error.message || 'Failed to assign schedule to agent', 400);
  }
};

const validateCreateSchedule = [
  body('customerId').notEmpty().withMessage('Customer is required'),
  body('customerName').notEmpty().withMessage('Customer name is required'),
  body('workOrderNo').notEmpty().withMessage('Work order number is required'),
  body('managerId').notEmpty().withMessage('Manager is required'),
  body('managerName').notEmpty().withMessage('Manager name is required'),
  body('frequency').isIn(['DAILY', 'WEEKLY', 'MONTHLY', 'QUARTERLY', 'YEARLY']).withMessage('Valid frequency is required'),
  body('periodFrom').isISO8601().withMessage('Valid period from date is required'),
  body('periodTo').isISO8601().withMessage('Valid period to date is required'),
];

module.exports = {
  getSchedules,
  getAgentSchedules,
  getCustomerSchedules,
  getScheduleById,
  createSchedule,
  generateJobs,
  updateSchedule,
  deleteSchedule,
  getScheduleJobs,
  assignScheduleToAgent,
  assignAgentToScheduleDate,
  validateCreateSchedule,
};


