/**
 * Media Controller
 */

const mediaService = require('../services/media.service');
const { successResponse, errorResponse } = require('../utils/response');

const uploadMedia = async (req, res) => {
  try {
    if (!req.file) {
      return errorResponse(res, 'No file uploaded', 400);
    }

    const { jobId } = req.body;
    const metadata = {
      caption: req.body.caption,
      isBeforePhoto: req.body.isBeforePhoto === 'true',
      isAfterPhoto: req.body.isAfterPhoto === 'true',
    };

    const media = await mediaService.uploadFile(
      req.file,
      jobId,
      req.tenantId,
      req.userId,
      metadata
    );

    return successResponse(res, media, 'File uploaded successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to upload file', 400);
  }
};

const uploadMultiple = async (req, res) => {
  try {
    if (!req.files || req.files.length === 0) {
      return errorResponse(res, 'No files uploaded', 400);
    }

    const { jobId } = req.body;
    const mediaFiles = await mediaService.uploadMultipleFiles(
      req.files,
      jobId,
      req.tenantId,
      req.userId
    );

    return successResponse(res, mediaFiles, 'Files uploaded successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to upload files', 400);
  }
};

const getMediaByJobId = async (req, res) => {
  try {
    const { jobId } = req.params;
    const media = await mediaService.getMediaByJobId(jobId, req.tenantId);
    return successResponse(res, media, 'Media retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve media', 500);
  }
};

const deleteMedia = async (req, res) => {
  try {
    const { id } = req.params;
    const result = await mediaService.deleteMedia(id, req.tenantId, req.userId);
    return successResponse(res, result, 'Media deleted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to delete media', 400);
  }
};

module.exports = {
  uploadMedia,
  uploadMultiple,
  getMediaByJobId,
  deleteMedia,
};

