/**
 * KPI Matrix Controller
 */

const KPIMatrix = require('../models/KPIMatrix.model');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult, query } = require('express-validator');
const logger = require('../utils/logger');

/**
 * Get KPI Matrix list
 */
const getKPIMatrices = async (req, res) => {
  try {
    const filters = {
      contractor: req.query.contractor,
      site: req.query.site,
      month: req.query.month,
      year: req.query.year,
      status: req.query.status,
      page: req.query.page,
      limit: req.query.limit,
    };

    const query = { tenantId: req.tenantId };

    if (filters.contractor) {
      query.contractor = { $regex: filters.contractor, $options: 'i' };
    }
    if (filters.site) {
      query.site = { $regex: filters.site, $options: 'i' };
    }
    if (filters.month) {
      query.month = parseInt(filters.month);
    }
    if (filters.year) {
      query.year = parseInt(filters.year);
    }
    if (filters.status) {
      query.status = filters.status;
    }

    const page = parseInt(filters.page) || 1;
    const limit = parseInt(filters.limit) || 20;
    const skip = (page - 1) * limit;

    const total = await KPIMatrix.countDocuments(query);
    const matrices = await KPIMatrix.find(query)
      .populate('createdBy', 'name email')
      .populate('updatedBy', 'name email')
      .sort({ year: -1, month: -1, createdAt: -1 })
      .skip(skip)
      .limit(limit);

    return successResponse(res, {
      matrices,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      },
    }, 'KPI Matrices retrieved successfully');
  } catch (error) {
    logger.error('Get KPI matrices error:', error);
    return errorResponse(res, error.message || 'Failed to retrieve KPI matrices', 500);
  }
};

/**
 * Get KPI Matrix by ID
 */
const getKPIMatrixById = async (req, res) => {
  try {
    const { id } = req.params;
    const matrix = await KPIMatrix.findOne({ _id: id, tenantId: req.tenantId })
      .populate('createdBy', 'name email')
      .populate('updatedBy', 'name email');

    if (!matrix) {
      return errorResponse(res, 'KPI Matrix not found', 404);
    }

    return successResponse(res, matrix, 'KPI Matrix retrieved successfully');
  } catch (error) {
    logger.error('Get KPI matrix by ID error:', error);
    return errorResponse(res, error.message || 'Failed to retrieve KPI Matrix', 500);
  }
};

/**
 * Create KPI Matrix
 */
const createKPIMatrix = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const matrixData = {
      ...req.body,
      tenantId: req.tenantId,
      createdBy: req.userId,
    };

    const matrix = new KPIMatrix(matrixData);
    await matrix.save();

    return successResponse(res, matrix, 'KPI Matrix created successfully', 201);
  } catch (error) {
    logger.error('Create KPI matrix error:', error);
    return errorResponse(res, error.message || 'Failed to create KPI Matrix', 400);
  }
};

/**
 * Update KPI Matrix
 */
const updateKPIMatrix = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;
    const matrix = await KPIMatrix.findOne({ _id: id, tenantId: req.tenantId });

    if (!matrix) {
      return errorResponse(res, 'KPI Matrix not found', 404);
    }

    Object.assign(matrix, req.body);
    matrix.updatedBy = req.userId;
    await matrix.save();

    return successResponse(res, matrix, 'KPI Matrix updated successfully');
  } catch (error) {
    logger.error('Update KPI matrix error:', error);
    return errorResponse(res, error.message || 'Failed to update KPI Matrix', 400);
  }
};

/**
 * Delete KPI Matrix
 */
const deleteKPIMatrix = async (req, res) => {
  try {
    const { id } = req.params;
    const matrix = await KPIMatrix.findOne({ _id: id, tenantId: req.tenantId });

    if (!matrix) {
      return errorResponse(res, 'KPI Matrix not found', 404);
    }

    await matrix.deleteOne();

    return successResponse(res, null, 'KPI Matrix deleted successfully');
  } catch (error) {
    logger.error('Delete KPI matrix error:', error);
    return errorResponse(res, error.message || 'Failed to delete KPI Matrix', 400);
  }
};

const validateKPIMatrix = [
  body('contractor').notEmpty().withMessage('Contractor is required'),
  body('site').notEmpty().withMessage('Site is required'),
  body('month').isInt({ min: 1, max: 12 }).withMessage('Month must be between 1 and 12'),
  body('year').isInt({ min: 2020, max: 2100 }).withMessage('Year must be a valid year'),
];

module.exports = {
  getKPIMatrices,
  getKPIMatrixById,
  createKPIMatrix,
  updateKPIMatrix,
  deleteKPIMatrix,
  validateKPIMatrix,
};

