/**
 * Job Controller
 */

const jobService = require('../services/job.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult, query } = require('express-validator');

/**
 * Create job
 */
const createJob = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const job = await jobService.createJob(
      req.body,
      req.tenantId,
      req.userId
    );

    return successResponse(res, job, 'Job created successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create job', 400);
  }
};

/**
 * Get jobs
 */
const getJobs = async (req, res) => {
  try {
    const filters = {
      status: req.query.status,
      ticketType: req.query.ticketType,
      assignedAgentId: req.query.assignedAgentId,
      ticketNumber: req.query.ticketNumber,
      fromDate: req.query.fromDate,
      toDate: req.query.toDate,
      page: req.query.page,
      limit: req.query.limit,
    };

    const result = await jobService.getJobs(
      filters,
      req.tenantId,
      req.userRole,
      req.userId,
      req.roleName // Pass roleName from middleware (for Employee with roleName "Manager")
    );

    return successResponse(res, result, 'Jobs retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve jobs', 500);
  }
};

/**
 * Get job by ID
 */
const getJobById = async (req, res) => {
  try {
    const { id } = req.params;
    
    // Use the dedicated getJobById function for better performance and proper access control
    const job = await jobService.getJobById(id, req.tenantId, req.userRole, req.userId, req.roleName);

    // Mark job as opened if agent is viewing it
    if ((req.userRole === 'SERVICE_AGENT' || req.userRole === 'EMPLOYEE') && 
        job.assignedAgentId?.toString() === req.userId.toString()) {
      await jobService.markJobOpened(id, req.userId, req.tenantId, req.userRole);
    }

    return successResponse(res, job, 'Job retrieved successfully');
  } catch (error) {
    // If error message is "Job not found", return 404, otherwise 500
    const statusCode = error.message === 'Job not found' ? 404 : 500;
    return errorResponse(res, error.message || 'Failed to retrieve job', statusCode);
  }
};

/**
 * Accept job
 */
const acceptJob = async (req, res) => {
  try {
    const { id } = req.params;
    const { timestamp } = req.body; // Get timestamp from mobile device

    if (req.userRole !== 'SERVICE_AGENT' && req.userRole !== 'EMPLOYEE') {
      return errorResponse(res, 'Only service agents or employees can accept jobs', 403);
    }

    const job = await jobService.acceptJob(id, req.userId, req.tenantId, timestamp);

    return successResponse(res, job, 'Job accepted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to accept job', 400);
  }
};

/**
 * Start job
 */
const startJob = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;
    const { location, timestamp } = req.body; // Get timestamp from mobile device

    if (req.userRole !== 'SERVICE_AGENT' && req.userRole !== 'EMPLOYEE') {
      return errorResponse(res, 'Only service agents or employees can start jobs', 403);
    }

    const job = await jobService.startJob(id, req.userId, req.tenantId, location, timestamp, req.userRole, req.roleName);

    return successResponse(res, job, 'Job started successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to start job', 400);
  }
};

/**
 * Submit job
 */
const submitJob = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;
    const { timestamp } = req.body; // Get timestamp from mobile device

    if (req.userRole !== 'SERVICE_AGENT' && req.userRole !== 'EMPLOYEE') {
      return errorResponse(res, 'Only service agents or employees can submit jobs', 403);
    }

    const job = await jobService.submitJob(id, req.userId, req.tenantId, { ...req.body, timestamp }, req.userRole, req.roleName);

    return successResponse(res, job, 'Job submitted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to submit job', 400);
  }
};

/**
 * Validation rules
 */
const validateStartJob = [
  body('location.lat').isFloat().withMessage('Valid latitude is required'),
  body('location.lng').isFloat().withMessage('Valid longitude is required'),
];

/**
 * Arrive at job location
 */
const arriveJob = async (req, res) => {
  try {
    const { id } = req.params;
    const { location, timestamp } = req.body; // Get timestamp from mobile device

    if (req.userRole !== 'SERVICE_AGENT' && req.userRole !== 'EMPLOYEE') {
      return errorResponse(res, 'Only agents or employees can arrive at jobs', 403);
    }

    const job = await jobService.arriveJob(id, req.userId, req.tenantId, location, req.userRole, timestamp, req.roleName);

    return successResponse(res, job, 'Arrived at job location successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to arrive at job', 400);
  }
};

/**
 * Update job status
 */
const updateJobStatus = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;
    const { status, description, photos, timestamp, monitoringRequired, signatureUrl, signatureData, signeeName } = req.body; // Get timestamp from mobile device

    if (req.userRole !== 'SERVICE_AGENT' && req.userRole !== 'EMPLOYEE') {
      return errorResponse(res, 'Only agents or employees can update job status', 403);
    }

    const job = await jobService.updateJobStatus(
      id,
      req.userId,
      req.tenantId,
      status,
      description,
      photos,
      req.userRole,
      timestamp,
      monitoringRequired,
      signatureUrl,
      signatureData,
      signeeName,
      req.roleName
    );

    return successResponse(res, job, 'Job status updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update job status', 400);
  }
};

/**
 * Get job timeline/activities
 */
const getJobTimeline = async (req, res) => {
  try {
    const { id } = req.params;

    const timeline = await jobService.getJobTimeline(id, req.tenantId);

    return successResponse(res, timeline, 'Job timeline retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve job timeline', 500);
  }
};

/**
 * Get pending jobs count
 */
const getPendingJobsCount = async (req, res) => {
  try {
    if (req.userRole !== 'SERVICE_AGENT' && req.userRole !== 'EMPLOYEE') {
      return errorResponse(res, 'Only agents can view pending jobs count', 403);
    }

    const count = await jobService.getPendingJobsCount(req.userId, req.tenantId);

    return successResponse(res, { count }, 'Pending jobs count retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve pending jobs count', 500);
  }
};

const validateSubmitJob = [
  body('workNotes').notEmpty().withMessage('Work notes are required'),
  body('workCompletedFully').isBoolean().withMessage('Work completion status is required'),
];

const validateUpdateJobStatus = [
  body('status').notEmpty().withMessage('Status is required'),
  body('status').isIn(['NEW', 'ACCEPTED', 'IN_PROGRESS', 'PENDING_APPROVAL', 'APPROVED', 'REJECTED', 'CLOSED', 'CANCELLED']).withMessage('Invalid status'),
  body('photos').optional().isArray().withMessage('Photos must be an array'),
];

/**
 * Close job by manager
 */
const closeJobByManager = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { closureDescription } = req.body;
    const job = await jobService.closeJobByManager(
      req.params.id,
      req.userId,
      req.tenantId,
      closureDescription,
      req.userRole, // Pass userRole for Employee manager detection
      req.roleName  // Pass roleName for Employee manager detection
    );

    return successResponse(res, job, 'Job closed successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to close job', 400);
  }
};

const validateCloseJob = [
  body('closureDescription').notEmpty().withMessage('Closure description is required'),
  body('closureDescription').isLength({ min: 10 }).withMessage('Closure description must be at least 10 characters'),
];

/**
 * Get unassigned jobs (jobs assigned to manager but not to any agent)
 */
const getUnassignedJobs = async (req, res) => {
  try {
    const filters = {
      status: req.query.status,
      ticketType: req.query.ticketType,
      fromDate: req.query.fromDate,
      toDate: req.query.toDate,
      page: req.query.page,
      limit: req.query.limit,
    };

    const result = await jobService.getUnassignedJobs(
      filters,
      req.tenantId,
      req.userRole,
      req.userId,
      req.roleName
    );

    return successResponse(res, result, 'Unassigned jobs retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve unassigned jobs', 500);
  }
};

/**
 * Get unassigned jobs for agent (jobs assigned to agent's manager but not to any agent)
 */
const getUnassignedJobsForAgent = async (req, res) => {
  try {
    const filters = {
      status: req.query.status,
      ticketType: req.query.ticketType,
      fromDate: req.query.fromDate,
      toDate: req.query.toDate,
      page: req.query.page,
      limit: req.query.limit,
    };

    const result = await jobService.getUnassignedJobsForAgent(
      filters,
      req.tenantId,
      req.userRole,
      req.userId,
      req.roleName
    );

    return successResponse(res, result, 'Unassigned jobs retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve unassigned jobs', 500);
  }
};

/**
 * Assign job to agent (by manager)
 */
const assignJobToAgent = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;
    const { agentId } = req.body;

    const job = await jobService.assignJobToAgent(
      id,
      agentId,
      req.userId,
      req.tenantId,
      req.userRole,
      req.roleName
    );

    return successResponse(res, job, 'Job assigned to agent successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to assign job to agent', 400);
  }
};

const validateAssignAgent = [
  body('agentId').notEmpty().withMessage('Agent ID is required'),
  body('agentId').isMongoId().withMessage('Invalid agent ID format'),
];

/**
 * Agent self-assignment (agent can take jobs assigned to their manager)
 */
const selfAssignJob = async (req, res) => {
  try {
    const { id } = req.params;

    const job = await jobService.selfAssignJob(
      id,
      req.userId,
      req.tenantId,
      req.userRole,
      req.roleName
    );

    return successResponse(res, job, 'Job self-assigned successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to self-assign job', 400);
  }
};

/**
 * Get pending signature jobs
 */
const getPendingSignatureJobs = async (req, res) => {
  try {
    const filters = {
      ticketType: req.query.ticketType,
      fromDate: req.query.fromDate,
      toDate: req.query.toDate,
      page: req.query.page,
      limit: req.query.limit,
    };

    const result = await jobService.getPendingSignatureJobs(
      filters,
      req.tenantId,
      req.userRole,
      req.userId,
      req.roleName
    );

    return successResponse(res, result, 'Pending signature jobs retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve pending signature jobs', 500);
  }
};

/**
 * Sign job (for managers and customers)
 */
const signJob = async (req, res) => {
  try {
    const { id } = req.params;
    const { signatureUrl, signatureData, signeeName, timestamp } = req.body;

    if (req.userRole !== 'MANAGER' && req.userRole !== 'CUSTOMER' && 
        !(req.userRole === 'EMPLOYEE' && req.roleName && req.roleName.toLowerCase() === 'manager')) {
      return errorResponse(res, 'Only managers and customers can sign jobs', 403);
    }

    if (!signatureUrl && !signatureData) {
      return errorResponse(res, 'Signature is required', 400);
    }

    const job = await jobService.signJob(
      id,
      req.userId,
      req.tenantId,
      { signatureUrl, signatureData, signeeName },
      req.userRole,
      timestamp,
      req.roleName
    );

    return successResponse(res, job, 'Job signed successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to sign job', 400);
  }
};

module.exports = {
  createJob,
  getJobs,
  getJobById,
  acceptJob,
  startJob,
  submitJob,
  arriveJob,
  updateJobStatus,
  getJobTimeline,
  getPendingJobsCount,
  validateStartJob,
  validateSubmitJob,
  validateUpdateJobStatus,
  closeJobByManager,
  validateCloseJob,
  getUnassignedJobs,
  getUnassignedJobsForAgent,
  assignJobToAgent,
  validateAssignAgent,
  selfAssignJob,
  getPendingSignatureJobs,
  signJob,
};

