/**
 * Employee Controller
 */

const employeeService = require('../services/employee.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult } = require('express-validator');

/**
 * Get employees
 */
const getEmployees = async (req, res) => {
  try {
    const filters = {
      status: req.query.status || undefined,
      role: req.query.role || undefined,
      roleName: req.query.roleName || undefined, // Add roleName filter
      search: req.query.search || undefined,
    };

    const employees = await employeeService.getEmployees(req.tenantId, filters);

    return successResponse(res, employees, 'Employees retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve employees', 500);
  }
};

/**
 * Get employee by ID
 */
const getEmployeeById = async (req, res) => {
  try {
    const { id } = req.params;

    const employee = await employeeService.getEmployeeById(id, req.tenantId);

    return successResponse(res, employee, 'Employee retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve employee', 404);
  }
};

/**
 * Create employee
 */
const createEmployee = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const employee = await employeeService.createEmployee(
      req.body,
      req.tenantId,
      req.userId // Company admin who created this employee
    );

    return successResponse(res, employee, 'Employee created successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create employee', 400);
  }
};

/**
 * Update employee
 */
const updateEmployee = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;

    // Validate password if provided
    if (req.body.password && req.body.password.trim() !== '') {
      if (req.body.password.length < 8) {
        return errorResponse(res, 'Password must be at least 8 characters long', 400);
      }
      if (!/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/.test(req.body.password)) {
        return errorResponse(res, 'Password must contain at least one uppercase letter, one lowercase letter, and one number', 400);
      }
    }

    const employee = await employeeService.updateEmployee(id, req.body, req.tenantId);

    return successResponse(res, employee, 'Employee updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update employee', 400);
  }
};

/**
 * Delete employee
 */
const deleteEmployee = async (req, res) => {
  try {
    const { id } = req.params;

    const result = await employeeService.deleteEmployee(id, req.tenantId);

    return successResponse(res, result, 'Employee deleted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to delete employee', 400);
  }
};

/**
 * Assign clients to employee
 */
const assignClients = async (req, res) => {
  try {
    const { id } = req.params;
    const { clientIds } = req.body;

    if (!Array.isArray(clientIds)) {
      return errorResponse(res, 'clientIds must be an array', 400);
    }

    const employee = await employeeService.assignClients(id, clientIds, req.tenantId);

    return successResponse(res, employee, 'Clients assigned successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to assign clients', 400);
  }
};

/**
 * Assign jobs to employee
 */
const assignJobs = async (req, res) => {
  try {
    const { id } = req.params;
    const { jobIds } = req.body;

    if (!Array.isArray(jobIds)) {
      return errorResponse(res, 'jobIds must be an array', 400);
    }

    const employee = await employeeService.assignJobs(id, jobIds, req.tenantId);

    return successResponse(res, employee, 'Jobs assigned successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to assign jobs', 400);
  }
};

/**
 * Create User account for employee (for Manager role)
 */
const createUserAccount = async (req, res) => {
  try {
    const { id } = req.params;

    const user = await employeeService.createUserAccountForEmployee(id, req.tenantId);

    return successResponse(res, user, 'User account created successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create user account', 400);
  }
};

/**
 * Validation rules
 */
const validateCreateEmployee = [
  body('firstName').notEmpty().withMessage('First name is required'),
  body('lastName').notEmpty().withMessage('Last name is required'),
  body('email').isEmail().withMessage('Valid email is required'),
  body('contactNumber').notEmpty().withMessage('Contact number is required'),
  body('password')
    .notEmpty().withMessage('Password is required')
    .isLength({ min: 8 }).withMessage('Password must be at least 8 characters long')
    .matches(/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/).withMessage('Password must contain at least one uppercase letter, one lowercase letter, and one number'),
];

const validateUpdateEmployee = [
  body('firstName').optional().notEmpty().withMessage('First name cannot be empty'),
  body('lastName').optional().notEmpty().withMessage('Last name cannot be empty'),
  body('email').optional().isEmail().withMessage('Valid email is required'),
  body('contactNumber').optional().notEmpty().withMessage('Contact number cannot be empty'),
];

module.exports = {
  getEmployees,
  getEmployeeById,
  createEmployee,
  updateEmployee,
  deleteEmployee,
  assignClients,
  assignJobs,
  createUserAccount,
  validateCreateEmployee,
  validateUpdateEmployee,
};

