/**
 * Customer Controller
 */

const customerService = require('../services/customer.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult } = require('express-validator');

/**
 * Get customers
 */
const getCustomers = async (req, res) => {
  try {
    const filters = {
      status: req.query.status || undefined,
      search: req.query.search || undefined,
    };

    const customers = await customerService.getCustomers(req.tenantId, filters);

    return successResponse(res, customers, 'Customers retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve customers', 500);
  }
};

/**
 * Get customer by ID
 */
const getCustomerById = async (req, res) => {
  try {
    const { id } = req.params;

    const customer = await customerService.getCustomerById(id, req.tenantId);

    return successResponse(res, customer, 'Customer retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve customer', 404);
  }
};

/**
 * Create customer
 */
const createCustomer = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const customer = await customerService.createCustomer(
      req.body,
      req.tenantId,
      req.userId
    );

    return successResponse(res, customer, 'Customer created successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create customer', 400);
  }
};

/**
 * Update customer
 */
const updateCustomer = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;

    // Validate password if provided
    if (req.body.password && req.body.password.trim() !== '') {
      if (req.body.password.length < 8) {
        return errorResponse(res, 'Password must be at least 8 characters long', 400);
      }
      if (!/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/.test(req.body.password)) {
        return errorResponse(res, 'Password must contain at least one uppercase letter, one lowercase letter, and one number', 400);
      }
    }

    const customer = await customerService.updateCustomer(id, req.body, req.tenantId);

    return successResponse(res, customer, 'Customer updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update customer', 400);
  }
};

/**
 * Delete customer
 */
const deleteCustomer = async (req, res) => {
  try {
    const { id } = req.params;

    const result = await customerService.deleteCustomer(id, req.tenantId);

    return successResponse(res, result, 'Customer deleted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to delete customer', 400);
  }
};

/**
 * Validation rules
 */
const validateCreateCustomer = [
  body('name').notEmpty().withMessage('Customer name is required'),
  body('email').isEmail().withMessage('Valid email is required'),
  body('contactNumber').notEmpty().withMessage('Contact number is required'),
  body('password')
    .notEmpty().withMessage('Password is required')
    .isLength({ min: 8 }).withMessage('Password must be at least 8 characters long')
    .matches(/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/).withMessage('Password must contain at least one uppercase letter, one lowercase letter, and one number'),
];

const validateUpdateCustomer = [
  body('name').optional().notEmpty().withMessage('Customer name cannot be empty'),
  body('email').optional().isEmail().withMessage('Valid email is required'),
  body('contactNumber').optional().notEmpty().withMessage('Contact number cannot be empty'),
];

module.exports = {
  getCustomers,
  getCustomerById,
  createCustomer,
  updateCustomer,
  deleteCustomer,
  validateCreateCustomer,
  validateUpdateCustomer,
};

