/**
 * Complaint Photo Upload Controller
 */

const multer = require('multer');
const path = require('path');
const fs = require('fs').promises;
const { successResponse, errorResponse } = require('../utils/response');
const logger = require('../utils/logger');

// Configure multer for complaint photo uploads
const storage = multer.diskStorage({
  destination: async (req, file, cb) => {
    const uploadDir = 'uploads/complaints';
    try {
      await fs.mkdir(uploadDir, { recursive: true });
      cb(null, uploadDir);
    } catch (error) {
      cb(error);
    }
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    const ext = path.extname(file.originalname);
    cb(null, `complaint-${uniqueSuffix}${ext}`);
  },
});

const upload = multer({
  storage,
  limits: {
    fileSize: 5 * 1024 * 1024, // 5MB per image
  },
  fileFilter: (req, file, cb) => {
    const allowedTypes = /jpeg|jpg|png|gif|webp/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    
    // Check MIME type - be more lenient if extension is valid
    const mimetype = file.mimetype && (
      file.mimetype.startsWith('image/jpeg') ||
      file.mimetype.startsWith('image/jpg') ||
      file.mimetype.startsWith('image/png') ||
      file.mimetype.startsWith('image/gif') ||
      file.mimetype.startsWith('image/webp')
    );

    // If extension is valid, accept the file even if MIME type is missing
    if (extname) {
      if (mimetype || !file.mimetype) {
        return cb(null, true);
      } else {
        // Log warning but still accept if extension is valid
        logger.warn('File has valid extension but unexpected MIME type', {
          filename: file.originalname,
          mimetype: file.mimetype,
          extension: path.extname(file.originalname),
        });
        return cb(null, true);
      }
    } else {
      cb(new Error('Invalid file type. Only image files are allowed (jpeg, jpg, png, gif, webp).'));
    }
  },
});

/**
 * Upload single complaint photo
 */
const uploadComplaintPhoto = async (req, res) => {
  try {
    if (!req.file) {
      return errorResponse(res, 'No file uploaded', 400);
    }

    const fileUrl = `/uploads/complaints/${req.file.filename}`;
    
    logger.info('Complaint photo uploaded', {
      filename: req.file.filename,
      size: req.file.size,
      url: fileUrl,
    });

    return successResponse(res, { 
      photoUrl: fileUrl,
      filename: req.file.filename,
    }, 'Photo uploaded successfully', 201);
  } catch (error) {
    logger.error('Upload complaint photo error:', error);
    return errorResponse(res, error.message || 'Failed to upload photo', 400);
  }
};

/**
 * Upload multiple complaint photos
 */
const uploadMultipleComplaintPhotos = async (req, res) => {
  try {
    if (!req.files || req.files.length === 0) {
      return errorResponse(res, 'No files uploaded', 400);
    }

    const photoUrls = req.files.map(file => ({
      photoUrl: `/uploads/complaints/${file.filename}`,
      filename: file.filename,
    }));

    logger.info('Multiple complaint photos uploaded', {
      count: req.files.length,
    });

    return successResponse(res, { photos: photoUrls }, 'Photos uploaded successfully', 201);
  } catch (error) {
    logger.error('Upload multiple complaint photos error:', error);
    return errorResponse(res, error.message || 'Failed to upload photos', 400);
  }
};

module.exports = {
  upload,
  uploadComplaintPhoto: [upload.single('photo'), uploadComplaintPhoto],
  uploadMultipleComplaintPhotos: [upload.array('photos', 10), uploadMultipleComplaintPhotos],
};

