/**
 * Complaint Category Controller
 */

const complaintCategoryService = require('../services/complaintCategory.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult } = require('express-validator');

/**
 * Get all complaint categories
 */
const getCategories = async (req, res) => {
  try {
    const categories = await complaintCategoryService.getCategories(req.tenantId, req.query);
    return successResponse(res, categories, 'Categories retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve categories', 500);
  }
};

/**
 * Get category by ID
 */
const getCategoryById = async (req, res) => {
  try {
    const { id } = req.params;
    const category = await complaintCategoryService.getCategoryById(id, req.tenantId);
    return successResponse(res, category, 'Category retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve category', 404);
  }
};

/**
 * Create complaint category
 */
const createCategory = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const category = await complaintCategoryService.createCategory(
      req.body,
      req.tenantId,
      req.userId
    );

    return successResponse(res, category, 'Category created successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to create category', 400);
  }
};

/**
 * Update complaint category
 */
const updateCategory = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;
    const category = await complaintCategoryService.updateCategory(
      id,
      req.body,
      req.tenantId
    );

    return successResponse(res, category, 'Category updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update category', 400);
  }
};

/**
 * Delete complaint category
 */
const deleteCategory = async (req, res) => {
  try {
    const { id } = req.params;
    await complaintCategoryService.deleteCategory(id, req.tenantId);
    return successResponse(res, { id }, 'Category deleted successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to delete category', 400);
  }
};

/**
 * Validation rules
 */
const validateCreateCategory = [
  body('name').notEmpty().withMessage('Category name is required'),
  body('name').isLength({ min: 2, max: 100 }).withMessage('Category name must be between 2 and 100 characters'),
  body('priority').optional().isIn(['Critical', 'Non-Critical']).withMessage('Priority must be either Critical or Non-Critical'),
];

const validateUpdateCategory = [
  body('name').optional().isLength({ min: 2, max: 100 }).withMessage('Category name must be between 2 and 100 characters'),
  body('priority').optional().isIn(['Critical', 'Non-Critical']).withMessage('Priority must be either Critical or Non-Critical'),
];

module.exports = {
  getCategories,
  getCategoryById,
  createCategory,
  updateCategory,
  deleteCategory,
  validateCreateCategory,
  validateUpdateCategory,
};

