/**
 * Authentication Controller
 */

const authService = require('../services/auth.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult } = require('express-validator');

/**
 * Login controller
 */
const login = async (req, res) => {
  const logger = require('../utils/logger');
  
  try {
    logger.info('🔐 [AUTH] Login request received', {
      ip: req.ip,
      userAgent: req.get('user-agent'),
      origin: req.get('origin'),
      body: { email: req.body.email, password: '***' }
    });

    // Validation
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      logger.warn('❌ [AUTH] Validation failed', { errors: errors.array() });
      return validationErrorResponse(res, errors.array());
    }

    const { email, password } = req.body;
    
    logger.info('✅ [AUTH] Validation passed, attempting login', { email });

    const result = await authService.login(email, password);

    logger.info('✅ [AUTH] Login successful', { 
      email, 
      role: result.user?.role,
      userId: result.user?.id || result.user?._id 
    });

    return successResponse(res, result, 'Login successful');
  } catch (error) {
    logger.error('❌ [AUTH] Login failed', { 
      error: error.message,
      stack: error.stack,
      email: req.body.email 
    });
    return errorResponse(res, error.message || 'Login failed', 401);
  }
};

/**
 * Refresh token controller
 */
const refreshToken = async (req, res) => {
  try {
    const { refreshToken } = req.body;

    if (!refreshToken) {
      return errorResponse(res, 'Refresh token required', 400);
    }

    const result = await authService.refreshToken(refreshToken);

    return successResponse(res, result, 'Token refreshed successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Token refresh failed', 401);
  }
};

/**
 * Logout controller
 */
const logout = async (req, res) => {
  try {
    // In a stateless JWT system, logout is handled client-side
    // But we can log the action for audit purposes
    return successResponse(res, null, 'Logout successful');
  } catch (error) {
    return errorResponse(res, 'Logout failed', 500);
  }
};

/**
 * Validation rules for login
 */
const validateLogin = [
  body('email').isEmail().withMessage('Valid email is required'),
  body('password').notEmpty().withMessage('Password is required'),
];

module.exports = {
  login,
  refreshToken,
  logout,
  validateLogin,
};

