/**
 * Attendance Controller
 */

const attendanceService = require('../services/attendance.service');
const { successResponse, errorResponse } = require('../utils/response');

/**
 * Get attendances
 */
const getAttendances = async (req, res) => {
  try {
    const filters = {
      employeeId: req.query.employeeId,
      startDate: req.query.startDate,
      endDate: req.query.endDate,
      date: req.query.date,
      status: req.query.status,
      page: req.query.page,
      limit: req.query.limit,
    };

    const result = await attendanceService.getAttendances(filters, req.tenantId);
    return successResponse(res, result, 'Attendances retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve attendances', 500);
  }
};

/**
 * Get attendance by ID
 */
const getAttendanceById = async (req, res) => {
  try {
    const { id } = req.params;
    const attendance = await attendanceService.getAttendanceById(id, req.tenantId);
    return successResponse(res, attendance, 'Attendance retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve attendance', 404);
  }
};

/**
 * Check-in
 */
const checkIn = async (req, res) => {
  try {
    const { employeeId, location, method } = req.body;
    const attendance = await attendanceService.checkIn(employeeId, req.tenantId, { location, method });
    return successResponse(res, attendance, 'Checked in successfully', 201);
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to check in', 400);
  }
};

/**
 * Check-out
 */
const checkOut = async (req, res) => {
  try {
    const { employeeId, location, method } = req.body;
    const attendance = await attendanceService.checkOut(employeeId, req.tenantId, { location, method });
    return successResponse(res, attendance, 'Checked out successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to check out', 400);
  }
};

/**
 * Add break
 */
const addBreak = async (req, res) => {
  try {
    const { employeeId, type, location } = req.body;
    const attendance = await attendanceService.addBreak(employeeId, req.tenantId, { type, location });
    return successResponse(res, attendance, 'Break started successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to start break', 400);
  }
};

/**
 * End break
 */
const endBreak = async (req, res) => {
  try {
    const { employeeId, breakIndex } = req.body;
    const attendance = await attendanceService.endBreak(employeeId, req.tenantId, breakIndex);
    return successResponse(res, attendance, 'Break ended successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to end break', 400);
  }
};

/**
 * Update attendance
 */
const updateAttendance = async (req, res) => {
  try {
    const { id } = req.params;
    const attendance = await attendanceService.updateAttendance(id, req.tenantId, req.body);
    return successResponse(res, attendance, 'Attendance updated successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to update attendance', 400);
  }
};

/**
 * Get attendance statistics
 */
const getAttendanceStats = async (req, res) => {
  try {
    const filters = {
      employeeId: req.query.employeeId,
      startDate: req.query.startDate,
      endDate: req.query.endDate,
    };

    const stats = await attendanceService.getAttendanceStats(req.tenantId, filters);
    return successResponse(res, stats, 'Attendance statistics retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve attendance statistics', 500);
  }
};

module.exports = {
  getAttendances,
  getAttendanceById,
  checkIn,
  checkOut,
  addBreak,
  endBreak,
  updateAttendance,
  getAttendanceStats,
};

