/**
 * Approval Controller
 */

const approvalService = require('../services/approval.service');
const { successResponse, errorResponse, validationErrorResponse } = require('../utils/response');
const { body, validationResult } = require('express-validator');

/**
 * Get pending approvals
 */
const getPendingApprovals = async (req, res) => {
  try {
    const approvals = await approvalService.getPendingApprovals(
      req.userId,
      req.tenantId
    );

    return successResponse(res, approvals, 'Pending approvals retrieved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to retrieve approvals', 500);
  }
};

/**
 * Approve job
 */
const approveJob = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;
    const { comments } = req.body;

    const job = await approvalService.approveJob(
      id,
      req.userId,
      req.tenantId,
      comments,
      req.userRole, // Pass userRole
      req.roleName // Pass roleName
    );

    return successResponse(res, job, 'Job approved successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to approve job', 400);
  }
};

/**
 * Reject job
 */
const rejectJob = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return validationErrorResponse(res, errors.array());
    }

    const { id } = req.params;
    const { rejectionReason, comments } = req.body;

    const job = await approvalService.rejectJob(
      id,
      req.userId,
      req.tenantId,
      rejectionReason,
      comments,
      req.userRole, // Pass userRole
      req.roleName // Pass roleName
    );

    return successResponse(res, job, 'Job rejected successfully');
  } catch (error) {
    return errorResponse(res, error.message || 'Failed to reject job', 400);
  }
};

/**
 * Validation rules
 */
const validateApprove = [
  body('comments').optional().isString(),
];

const validateReject = [
  body('rejectionReason').isIn(['PHOTOS_NOT_CLEAR', 'INCOMPLETE_WORK', 'MISSING_SIGNATURE', 'POOR_QUALITY', 'OTHER']).withMessage('Valid rejection reason is required'),
  body('comments').optional().isString(),
];

module.exports = {
  getPendingApprovals,
  approveJob,
  rejectJob,
  validateApprove,
  validateReject,
};

