/**
 * HRQ API Configuration
 * Centralized API endpoints configuration
 */

const API_CONFIG = {
  // Base URL - Update this for different environments
  BASE_URL: process.env.API_BASE_URL || 'http://localhost:3000',
  
  // API Version
  API_VERSION: 'v1',
  
  // Endpoints
  ENDPOINTS: {
    // Authentication
    AUTH: {
      LOGIN: '/auth/login',
      REFRESH_TOKEN: '/auth/refresh-token',
      LOGOUT: '/auth/logout',
      REGISTER: '/auth/register',
      FORGOT_PASSWORD: '/auth/forgot-password',
      RESET_PASSWORD: '/auth/reset-password',
      VERIFY_EMAIL: '/auth/verify-email',
    },
    
    // Users
    USERS: {
      BASE: '/users',
      LIST: '/users',
      CREATE: '/users',
      GET_BY_ID: (id) => `/users/${id}`,
      UPDATE: (id) => `/users/${id}`,
      DELETE: (id) => `/users/${id}`,
      GET_PROFILE: '/users/profile',
      UPDATE_PROFILE: '/users/profile',
      CHANGE_PASSWORD: '/users/change-password',
    },
    
    // Tenants
    TENANTS: {
      BASE: '/tenants',
      LIST: '/tenants',
      CREATE: '/tenants',
      GET_BY_ID: (id) => `/tenants/${id}`,
      UPDATE: (id) => `/tenants/${id}`,
      DELETE: (id) => `/tenants/${id}`,
      WHITE_LABEL_CONFIG: (id) => `/tenants/${id}/white-label`,
    },
    
    // Jobs
    JOBS: {
      BASE: '/jobs',
      LIST: '/jobs',
      CREATE: '/jobs',
      GET_BY_ID: (id) => `/jobs/${id}`,
      UPDATE: (id) => `/jobs/${id}`,
      DELETE: (id) => `/jobs/${id}`,
      ACCEPT: (id) => `/jobs/${id}/accept`,
      START: (id) => `/jobs/${id}/start`,
      SUBMIT: (id) => `/jobs/${id}/submit`,
      COMPLETE: (id) => `/jobs/${id}/complete`,
      CANCEL: (id) => `/jobs/${id}/cancel`,
      GET_MY_JOBS: '/jobs/my-jobs',
      GET_PENDING: '/jobs/pending',
      GET_BY_STATUS: (status) => `/jobs?status=${status}`,
    },
    
    // Approvals
    APPROVALS: {
      BASE: '/approvals',
      PENDING: '/approvals/pending',
      APPROVE: (id) => `/approvals/${id}/approve`,
      REJECT: (id) => `/approvals/${id}/reject`,
      GET_BY_JOB_ID: (jobId) => `/approvals/job/${jobId}`,
    },
    
    // Media
    MEDIA: {
      BASE: '/media',
      UPLOAD: '/media/upload',
      UPLOAD_MULTIPLE: '/media/upload-multiple',
      GET_BY_ID: (id) => `/media/${id}`,
      DELETE: (id) => `/media/${id}`,
      GET_BY_JOB_ID: (jobId) => `/media/job/${jobId}`,
    },
    
    // Reports
    REPORTS: {
      BASE: '/reports',
      GENERATE: (jobId) => `/reports/generate/${jobId}`,
      GET_BY_ID: (id) => `/reports/${id}`,
      GET_BY_JOB_ID: (jobId) => `/reports/job/${jobId}`,
      DOWNLOAD: (id) => `/reports/${id}/download`,
      LIST: '/reports',
    },
    
    // Periodic Maintenance
    PERIODIC_MAINTENANCE: {
      BASE: '/periodic-maintenance',
      SCHEDULES: '/periodic-maintenance/schedules',
      CREATE_SCHEDULE: '/periodic-maintenance/schedules',
      UPDATE_SCHEDULE: (id) => `/periodic-maintenance/schedules/${id}`,
      DELETE_SCHEDULE: (id) => `/periodic-maintenance/schedules/${id}`,
      GENERATE_JOBS: (scheduleId) => `/periodic-maintenance/schedules/${scheduleId}/generate-jobs`,
      GET_SCHEDULE_JOBS: (scheduleId) => `/periodic-maintenance/schedules/${scheduleId}/jobs`,
    },
    
    // Compliance Management
    COMPLIANCE: {
      BASE: '/compliance',
      CREATE_COMPLAINT: '/compliance/complaints',
      LIST_COMPLAINTS: '/compliance/complaints',
      GET_COMPLAINT: (id) => `/compliance/complaints/${id}`,
      UPDATE_COMPLAINT: (id) => `/compliance/complaints/${id}`,
      AUTO_ASSIGN: (id) => `/compliance/complaints/${id}/auto-assign`,
    },
    
    // KPI
    KPI: {
      BASE: '/kpi',
      GET_USER_KPI: (userId) => `/kpi/${userId}`,
      GET_TEAM_KPI: '/kpi/team',
      GET_ORG_KPI: '/kpi/organization',
      LEADERBOARD: '/kpi/leaderboard',
      GET_HISTORY: (userId) => `/kpi/${userId}/history`,
    },
    
    // SLA
    SLA: {
      BASE: '/sla',
      MONITOR: '/sla/monitor',
      GET_BREACHES: '/sla/breaches',
      ESCALATE: (jobId) => `/sla/jobs/${jobId}/escalate`,
      GET_JOB_SLA: (jobId) => `/sla/jobs/${jobId}`,
    },
    
    // Notifications
    NOTIFICATIONS: {
      BASE: '/notifications',
      LIST: '/notifications',
      GET_BY_ID: (id) => `/notifications/${id}`,
      MARK_READ: (id) => `/notifications/${id}/read`,
      MARK_ALL_READ: '/notifications/read-all',
      UNREAD_COUNT: '/notifications/unread-count',
    },
    
    // Audit Logs
    AUDIT_LOGS: {
      BASE: '/audit-logs',
      LIST: '/audit-logs',
      GET_BY_ID: (id) => `/audit-logs/${id}`,
      GET_BY_ENTITY: (entityType, entityId) => `/audit-logs/${entityType}/${entityId}`,
    },
    
    // Dashboard
    DASHBOARD: {
      BASE: '/dashboard',
      STATS: '/dashboard/stats',
      AGENT_DASHBOARD: '/dashboard/agent',
      MANAGER_DASHBOARD: '/dashboard/manager',
      ADMIN_DASHBOARD: '/dashboard/admin',
      SUPER_ADMIN_DASHBOARD: '/dashboard/super-admin',
    },
  },
  
  // Helper function to get full URL
  getFullUrl: function(endpoint) {
    return `${this.BASE_URL}/api/${this.API_VERSION}${endpoint}`;
  },
  
  // Helper function to get endpoint
  getEndpoint: function(category, action, ...params) {
    const endpoint = this.ENDPOINTS[category]?.[action];
    if (typeof endpoint === 'function') {
      return endpoint(...params);
    }
    return endpoint;
  },
};

module.exports = API_CONFIG;

